// Here we implement our first dynamic algorithm for dynamic non-monotone submodular.
// In this code, you can see run_id and bucket_id in argument of functions.
// The run_id refer to two instance of dynamic monotone submodular. When run_id = 0,
// it refers to instance I_1 and when run_id = 1, it refers to instance I_2.
// The bucket_id refer to different guess of OPT. The value of (1+epsilon)^bucket_it
// is our guess for the OPT.
#include<bits/stdc++.h>
#include "oracle.h"
using namespace std;
typedef long double ld;
// Here we define epsilon and the number of different guess for OPT
long double epsilon = 1;
const int bucket_number = 20;
// Here we define variables.
// As you can see some variables have [2] which refers to two different value of run_id.
// Also some variable have [bucket_number] which means that we keep different value
// for these variables for different guesses of OPT.
// The variable s[][][] refers to seleceted element at each level.
// Variable fs[][][i] refers to f(s[][][0..i]).
// Variable ans refers to S1 for each run and fsprime1 refers to S'_1.
// For each element, the start_bucket and end_bucket define in which interval of bucket
// this element will be added.
// The value OPT keep our guess of OPT in different buckets, i.e. OPT[i] := (1+epsilon)^i.
// Variable r[][][i] refers to elements remain at level i.
// Finally, variable max_cut mainatain all S_1, S'_1 and S_2 for all runs and buckets.
const int maxn = 200, maxk = 10;
int k, s[2][bucket_number][maxk], start_bucket[maxn], end_bucket[maxn];
ld OPT[bucket_number], fs[2][bucket_number][maxk], ans[2][bucket_number], fsprime1[bucket_number];
set<int> r[2][bucket_number][maxk];
multiset<ld> max_cut;
vector<int> sprime1[bucket_number];

void remove_from_bucket(int, int, int);
void add_to_bucket(int, int, int);

// Return true with probability 1/p
bool select_with_probability(int p)
{
  return (rand() % p == 0);
}

// Select an element from elements uniformely random
int select_one_element(set<int> elements)
{
  int random_idx = rand() % elements.size();
  set<int>::iterator it = elements.begin();
  advance(it, random_idx);
  return *it;
}

// Return max element of max_cut or return 0 if its empty
ld get_answer()
{
  if(max_cut.size() == 0)
    return 0;
  return (*(--max_cut.end()));
}

// Change a value in max_cut from prev_value to new_value
ld change_max_cut(int prev_value, int new_value)
{
  multiset<ld>::iterator mc_it = max_cut.find(prev_value);
  if(mc_it != max_cut.end())
    max_cut.erase(mc_it);
  max_cut.insert(new_value);
  return new_value;
}

// Filter elements in r[run_id][bucket_id][idx] and save filtered elements in r[run_id][bucket_id][idx+1]
// g is a vector which keep selected elements until now and selected_element is last element of g.
// We should eliminate selected_element from r
void filter(int run_id, int bucket_id, int idx, vector<int> g, int selected_element)
{
  r[run_id][bucket_id][idx + 1].clear();
  ld acceptance_marginal = fs[run_id][bucket_id][idx + 1] + OPT[bucket_id] / (5 * k);
  for(set<int>::iterator r_it = r[run_id][bucket_id][idx].begin(); r_it != r[run_id][bucket_id][idx].end(); r_it ++)
    {
      int element = *r_it;
      g.push_back(element);
      ld additive_value = f(g) - fs[run_id][bucket_id][idx + 1];
      g.pop_back();
      if(element !=  selected_element && additive_value >= acceptance_marginal)
	r[run_id][bucket_id][idx + 1].insert(element);
    }
}

// Here, we restart algorithm for instance (run_id, bucket_id) after index idx.
// Vector g maintain selected elements until now.
// The force_add is keeping the next element we are going to select. This value
// can be 0 which means we are not forced to select specified value.
void re_run(int run_id, int bucket_id, int idx, vector<int> g, int force_add)
{
  vector<int> old_s, new_s, old_elements(k), new_elements(k);
  // Remove old value that selected
  for(int i = idx + 1; i <= k; i ++)
    if(s[run_id][bucket_id][i] != 0)
      {
	old_s.push_back(s[run_id][bucket_id][i]);
	s[run_id][bucket_id][i] = 0;
	fs[run_id][bucket_id][i] = 0;
      }
  // For i = idx..k-1, select s[i+1]
  for(int i = idx; i < k; i ++)
    {
      if (r[run_id][bucket_id][i].size() == 0)
	break;
      // The next_element is the element we select at this level 
      int next_element = force_add;
      if(i > idx || next_element == 0)
	next_element = select_one_element(r[run_id][bucket_id][i]);
      new_s.push_back(next_element);
      s[run_id][bucket_id][i + 1] = next_element;
      g.push_back(next_element);
      fs[run_id][bucket_id][i + 1] = f(g);
      // Filter r based on the selected element
      if(i + 1 < k)
	filter(run_id, bucket_id, i, g, next_element);
    }
  // Update max_cut based on new selected elements for this bucket
  ans[run_id][bucket_id] = change_max_cut(ans[run_id][bucket_id], fs[run_id][bucket_id][g.size()]);
  // If we are at instance I_1, we also should compute S'_1 and S_2
  if(run_id == 0)
    {
      vector<int>::iterator vit;
      // Add elements to I_2 that previously was selected in S_1 but not now
      vit = set_difference(old_s.begin(), old_s.end(), new_s.begin(), new_s.end(), old_elements.begin());
      old_elements.resize(vit - old_elements.begin());
      for(int i = 0; i < old_elements.size(); i ++)
	add_to_bucket(1, bucket_id, old_elements[i]);
      // Remove new selected elements from I_2
      vit = set_difference(new_s.begin(), new_s.end(), old_s.begin(), old_s.end(), new_elements.begin());
      new_elements.resize(vit - new_elements.begin());
      for(int i = 0; i < new_elements.size(); i ++)
	remove_from_bucket(1, bucket_id, new_elements[i]);
      // Update S'_1
      sprime1[bucket_id].clear();
      for(int i = 1; i <= k; i ++)
	if(s[run_id][bucket_id][i] != 0 && select_with_probability(2))
	  sprime1[bucket_id].push_back(s[run_id][bucket_id][i]);
      fsprime1[bucket_id] = change_max_cut(fsprime1[bucket_id], f(sprime1[bucket_id]));
    }
}

// Remove element x from (run_id, bucket_id)
void remove_from_bucket(int run_id, int bucket_id, int x)
{
  vector<int> g;
  for(int i = 0; i < k; i ++)
    {
      set<int>::iterator r_it = r[run_id][bucket_id][i].find(x);
      if(r_it == r[run_id][bucket_id][i].end())
	return;
      r[run_id][bucket_id][i].erase(r_it);
      // If this element was selected at this step, we should restart after this step
      if(s[run_id][bucket_id][i + 1] == x)
	{
	  s[run_id][bucket_id][i + 1] = 0;
	  re_run(run_id, bucket_id, i, g, 0);
	  return;
	}
      g.push_back(s[run_id][bucket_id][i+1]);
    }
}

// Add element x to (run_id, bucket_id)
void add_to_bucket(int run_id, int bucket_id, int x)
{
  vector<int> g;
  for(int i = 0; i < k; i ++)
    {
      r[run_id][bucket_id][i].insert(x);
      // If we select this element with probability of 1/len(r), we should restart after this step
      if(select_with_probability(r[run_id][bucket_id][i].size()))
	{
	  re_run(run_id, bucket_id, i, g, x);
	  return;
	}
      g.push_back(s[run_id][bucket_id][i + 1]);
      // If x filtererd based on selected elements until this step, stop this function
      g.push_back(x);
      ld additive_value = f(g) - fs[run_id][bucket_id][i + 1];
      g.pop_back();
      if(additive_value < OPT[bucket_id] / (5 * k))
	return;
    }
}

// Remove element x from buckets [start_bucket[x], end_bucket[x])
ld remove_element(int x)
{
  for(int i = start_bucket[x]; i < end_bucket[x]; i ++)
    {
      remove_from_bucket(0, i, x);
      remove_from_bucket(1, i, x);
    }
  return get_answer();
}

// Add element x to buckets [start_bucket[x], end_bucket[x])
ld add_element(int x)
{
  // First, we find the interval of OPT that we are going to insert x
  vector<int> tmp(1);
  tmp[0] = x;
  ld f_x = f(tmp);
  start_bucket[x] = lower_bound(OPT, OPT + bucket_number, f_x) - OPT;
  end_bucket[x] = upper_bound(OPT, OPT + bucket_number, f_x * k / epsilon) - OPT;
  // Now for each bucket, first we add x to I_1, if it was not selected, we add it to I_2
  for(int i = start_bucket[x]; i < end_bucket[x]; i ++)
    {
      add_to_bucket(0, i, x);
      bool is_in_S1 = false;
      for(int j = 1; j <= k;j ++)
	if (s[0][i][j] == x)
	  is_in_S1 = true;
      if(!is_in_S1)
	add_to_bucket(1, i, x);
    }
  return get_answer();
}

// Initialize k and OPT
void init(int K)
{
  srand(5000);
  k = K;
  epsilon = (ld)min(4.0, k/2.0);
  OPT[0] = 1;
  for(int i = 1; i < bucket_number; i ++)
    OPT[i] = OPT[i-1] * (1 + epsilon);
}

void print_selected_elements()
{
  int I=0,J=0,K=0;
  for(int i = 0; i < 2; i ++)
    for(int j = 0; j < bucket_number; j ++)
      {
	if(ans[i][j] >= K)
	  I=i, J=j, K=ans[i][j];
	if(fsprime1[j] > K)
	  I=-1, J=j, K=fsprime1[j];
      }
  cout<<"!!!!!!! ";
  if(I == -1)
    for(int i = 0; i < sprime1[J].size(); i ++)
      cout << sprime1[J][i] << " ";
  else
    for(int k = 1; s[I][J][k] != 0; k ++)
      cout << s[I][J][k] << " ";
  cout << endl;
}
